//
//  PredictionCalculations.swift
//  death_app Watch App
//
//  Utility functions for prediction calculations
//

import Foundation

struct PredictionCalculations {
    
    /// Calculate compound hazard ratio using Cox proportional hazards model
    static func combineHazardRatios(_ ratios: [Double]) -> Double {
        let combined = ratios.reduce(1.0, *)
        return min(combined, 5.0) // Cap at 5x baseline risk
    }
    
    /// Apply hazard ratio to baseline life expectancy
    static func applyHazardRatio(baseline: Double, hazardRatio: Double) -> Double {
        // Higher hazard ratio = lower life expectancy
        // Using simplified formula: adjusted = baseline / hazardRatio^0.5
        let adjustment = pow(hazardRatio, 0.5)
        return baseline / adjustment
    }
    
    /// Calculate age-specific baseline adjustments
    static func getAgeBasedMultiplier(age: Int) -> Double {
        // Younger people have more time for risk factors to compound
        switch age {
        case 18...30: return 1.2  // Higher impact of risk factors
        case 31...45: return 1.1
        case 46...60: return 1.0  // Standard impact
        case 61...75: return 0.9  // Reduced impact
        default: return 0.8       // Lower impact in elderly
        }
    }
    
    /// Validate prediction bounds
    static func validatePrediction(_ prediction: Double, age: Int) -> Double {
        let minLifeExpectancy = Double(age) + 0.5 // At least 6 months
        let maxLifeExpectancy = Double(age) + 50.0 // No more than 50 years added
        
        return max(minLifeExpectancy, min(prediction, maxLifeExpectancy))
    }
    
    /// Calculate confidence interval for prediction
    static func calculateConfidenceInterval(prediction: Double, riskFactorCount: Int) -> (lower: Double, upper: Double) {
        // More risk factors = wider confidence interval
        let baseUncertainty = 2.0 // Base +/- 2 years
        let riskAdjustment = Double(riskFactorCount) * 0.5
        let totalUncertainty = baseUncertainty + riskAdjustment
        
        return (
            lower: max(0, prediction - totalUncertainty),
            upper: prediction + totalUncertainty
        )
    }
}

// MARK: - Research-Based Constants

struct ResearchConstants {
    
    // Heart Rate (Jensen 2013)
    static let heartRateBaselineRPM = 60.0
    static let heartRateRiskPer10BPM = 0.16
    
    // VO2 Max (Kodama 2009)
    static let vo2MaxPercentileRisks: [Double] = [
        1.5, 1.3, 1.15, 1.0, 0.85, 0.7 // From bottom 10% to top 10%
    ]
    
    // Smoking (Doll 2004)
    static let smokingBaselineHazardRatio = 1.7 // 10 cigs/day
    static let smokingRiskPerCigarette = 0.05
    static let smokingQuitRecoveryRate = 0.05 // 5% per year
    static let smokingMaxRecovery = 0.8 // 80% max recovery
    
    // BMI (Whitlock 2009)
    static let bmiOptimalRange = 18.5...25.0
    static let bmiRiskCurve = [
        (18.5, 1.4), // Underweight
        (25.0, 1.0), // Normal
        (30.0, 1.1), // Overweight
        (35.0, 1.3), // Obese I
        (40.0, 1.5), // Obese II
        (50.0, 1.8)  // Obese III
    ]
    
    // Sleep (Cappuccio 2010)
    static let sleepOptimalHours = 7.5
    static let sleepRiskPerHourDeviation = 0.05
    static let sleepMaxRisk = 1.4
    
    // Cox Model Parameters
    static let maxCombinedHazardRatio = 3.0
    static let minLifeExpectancyMultiplier = 0.3 // 30% of baseline minimum
}

// MARK: - Validation Helpers

struct ValidationHelpers {
    
    static func isValidHeartRate(_ hr: Double) -> Bool {
        return hr >= 40 && hr <= 200
    }
    
    static func isValidVO2Max(_ vo2: Double, age: Int) -> Bool {
        let minExpected = age > 60 ? 15 : 20
        let maxExpected = age < 30 ? 80 : 60
        return vo2 >= Double(minExpected) && vo2 <= Double(maxExpected)
    }
    
    static func isValidBMI(_ bmi: Double) -> Bool {
        return bmi >= 12 && bmi <= 60
    }
    
    static func isValidSleepHours(_ hours: Double) -> Bool {
        return hours >= 3 && hours <= 14
    }
    
    static func validateAndClamp<T: Comparable>(_ value: T, min: T, max: T) -> T {
        return Swift.max(min, Swift.min(value, max))
    }
}